<?php

namespace App\Http\Livewire\Admin;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\User;
use App\Models\TutorAssignment;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Auth;

class UserManagement extends Component
{
    use WithPagination;

    // Filtros
    public string $search = '';
    public string $filterRol = '';
    public string $filterEstado = '';
    public string $filterTutor = '';
    public int $perPage = 15;

    // Modal Create/Edit
    public bool $showModal = false;
    public bool $editMode = false;
    public ?int $editingUserId = null;

    // Form fields
    public string $nombre = '';
    public string $codigo = '';
    public string $email = '';
    public string $rol = '';
    public string $telefono = '';
    public bool $activo = true;

    // Modal asignación de tutor
    public bool $showAsignModal = false;
    public ?int $selectedTutoradoId = null;
    public string $selectedTutorId = '';

    // Modal reset password
    public bool $showResetModal = false;
    public ?int $resetUserId = null;
    public string $newPassword = '';
    public string $generatedPassword = '';

    protected $queryString = ['search', 'filterRol', 'filterEstado', 'filterTutor'];

    protected function rules(): array
    {
        $uniqueEmail = $this->editMode
            ? "unique:users,email,{$this->editingUserId}"
            : 'unique:users,email';
        $uniqueCodigo = $this->editMode
            ? "unique:users,codigo,{$this->editingUserId}"
            : 'unique:users,codigo';

        return [
            'nombre' => 'required|string|min:3|max:100',
            'codigo' => "nullable|string|max:20|{$uniqueCodigo}",
            'email' => "required|email|{$uniqueEmail}",
            'rol' => 'required|in:administrador,tutor,tutorado',
            'telefono' => 'nullable|string|max:20',
        ];
    }

    public function updatingSearch(): void
    {
        $this->resetPage();
    }

    public function updatingFilterRol(): void
    {
        $this->resetPage();
    }

    public function openCreateModal(): void
    {
        $this->resetForm();
        $this->editMode = false;
        $this->showModal = true;
    }

    public function openEditModal(int $userId): void
    {
        $user = User::findOrFail($userId);
        $this->editingUserId = $userId;
        $this->nombre = $user->name;
        $this->codigo = $user->codigo ?? '';
        $this->email = $user->email;
        $this->rol = $user->roles->first()?->name ?? '';
        $this->telefono = $user->telefono ?? '';
        $this->activo = $user->activo;
        $this->editMode = true;
        $this->showModal = true;
    }

    public function save(): void
    {
        $this->validate();

        if ($this->editMode) {
            $user = User::findOrFail($this->editingUserId);
            $user->update([
                'name' => $this->nombre,
                'codigo' => $this->codigo ?: null,
                'email' => $this->email,
                'telefono' => $this->telefono ?: null,
                'activo' => $this->activo,
            ]);

            // Actualizar rol si cambió
            $user->syncRoles([$this->rol]);

            activity()->causedBy(Auth::user())->performedOn($user)
                ->log("Usuario actualizado: {$user->name}");

            session()->flash('success', 'Usuario actualizado correctamente.');
        } else {
            $tempPassword = Str::random(10) . '!1';
            $user = User::create([
                'name' => $this->nombre,
                'codigo' => $this->codigo ?: null,
                'email' => $this->email,
                'password' => Hash::make($tempPassword),
                'telefono' => $this->telefono ?: null,
                'activo' => $this->activo,
            ]);

            $user->assignRole($this->rol);

            activity()->causedBy(Auth::user())->performedOn($user)
                ->log("Usuario creado: {$user->name} con rol {$this->rol}");

            session()->flash('success', "Usuario creado. Contraseña temporal: {$tempPassword}");
        }

        $this->showModal = false;
        $this->resetForm();
    }

    public function toggleActivo(int $userId): void
    {
        $user = User::findOrFail($userId);
        $user->update(['activo' => !$user->activo]);

        $estado = $user->activo ? 'activado' : 'desactivado';
        activity()->causedBy(Auth::user())->performedOn($user)
            ->log("Usuario {$estado}: {$user->name}");

        session()->flash('success', "Usuario {$estado} correctamente.");
    }

    public function openAsignModal(int $tutoradoId): void
    {
        $this->selectedTutoradoId = $tutoradoId;
        $this->selectedTutorId = '';
        $this->showAsignModal = true;
    }

    public function asignarTutor(): void
    {
        $this->validate([
            'selectedTutorId' => 'required|exists:users,id',
        ]);

        $tutorado = User::findOrFail($this->selectedTutoradoId);
        $tutor = User::findOrFail($this->selectedTutorId);

        // Desactivar asignación anterior
        TutorAssignment::where('tutorado_id', $this->selectedTutoradoId)
            ->update(['activo' => false]);

        // Crear nueva asignación
        TutorAssignment::create([
            'tutor_id' => $tutor->id,
            'tutorado_id' => $tutorado->id,
            'fecha_asignacion' => now(),
            'activo' => true,
            'assigned_by' => Auth::id(),
        ]);

        activity()->causedBy(Auth::user())
            ->log("Asignación: {$tutorado->name} → Tutor: {$tutor->name}");

        // Notificar al tutor
        $tutor->notify(new \App\Notifications\TutoradoAsignado($tutorado));

        session()->flash('success', "Tutorado asignado correctamente al tutor {$tutor->name}.");
        $this->showAsignModal = false;
    }

    public function openResetModal(int $userId): void
    {
        $this->resetUserId = $userId;
        $this->generatedPassword = '';
        $this->showResetModal = true;
    }

    public function generarPassword(): void
    {
        $this->generatedPassword = Str::random(8) . rand(10, 99) . '!';
    }

    public function confirmarReset(): void
    {
        $this->validate([
            'generatedPassword' => 'required|min:8',
        ]);

        $user = User::findOrFail($this->resetUserId);
        $user->update(['password' => Hash::make($this->generatedPassword)]);

        activity()->causedBy(Auth::user())->performedOn($user)
            ->log("Contraseña restablecida para: {$user->name}");

        session()->flash('success', "Contraseña restablecida. Nueva contraseña: {$this->generatedPassword}");
        $this->showResetModal = false;
    }

    public function deleteUser(int $userId): void
    {
        $user = User::findOrFail($userId);

        // No puede eliminarse a sí mismo
        if ($user->id === Auth::id()) {
            session()->flash('error', 'No puedes eliminar tu propia cuenta.');
            return;
        }

        activity()->causedBy(Auth::user())->performedOn($user)
            ->log("Usuario eliminado (soft delete): {$user->name}");

        $user->delete();
        session()->flash('success', 'Usuario eliminado correctamente.');
    }

    protected function resetForm(): void
    {
        $this->nombre = '';
        $this->codigo = '';
        $this->email = '';
        $this->rol = '';
        $this->telefono = '';
        $this->activo = true;
        $this->editingUserId = null;
        $this->resetErrorBag();
    }

    public function getTutoresProperty()
    {
        return User::tutores()->activos()->orderBy('name')->get();
    }

    public function render()
    {
        $users = User::query()
            ->when($this->search, function ($q) {
                $q->where(function ($q) {
                    $q->where('name', 'like', "%{$this->search}%")
                        ->orWhere('email', 'like', "%{$this->search}%")
                        ->orWhere('codigo', 'like', "%{$this->search}%");
                });
            })
            ->when($this->filterRol, fn($q) => $q->role($this->filterRol))
            ->when($this->filterEstado !== '', function ($q) {
                $q->where('activo', $this->filterEstado === '1');
            })
            ->when($this->filterTutor, function ($q) {
                $q->whereHas('tutorAsignado', fn($q) => $q->where('tutor_id', $this->filterTutor));
            })
            ->with(['roles', 'tutorAsignado.tutor'])
            ->orderBy('name')
            ->paginate($this->perPage);

        return view('livewire.admin.user-management', [
            'users' => $users,
            'tutores' => $this->tutores,
        ])->layout('layouts.app', ['title' => 'Gestión de Usuarios']);
    }
}
