<?php

namespace App\Http\Livewire\Tutor;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\TutoringSession;
use App\Models\TutorAssignment;
use App\Models\User;
use Illuminate\Support\Facades\Auth;

class SessionManager extends Component
{
    use WithPagination;

    public string $search = '';
    public string $filterTutorado = '';
    public string $filterEstado = '';
    public string $filterMes = '';

    // Modal
    public bool $showModal = false;
    public bool $editMode = false;
    public ?int $editingId = null;

    // Form
    public string $tutorado_id = '';
    public string $fecha = '';
    public string $hora_inicio = '';
    public string $hora_fin = '';
    public string $modalidad = 'presencial';
    public string $tipo = 'individual';
    public string $temas_tratados = '';
    public string $acuerdos = '';
    public string $compromisos_tutorado = '';
    public string $compromisos_tutor = '';
    public string $observaciones = '';
    public string $estado = 'realizada';
    public string $proxima_sesion = '';

    protected function rules(): array
    {
        return [
            'tutorado_id' => 'required|exists:users,id',
            'fecha' => 'required|date',
            'hora_inicio' => 'nullable|date_format:H:i',
            'hora_fin' => 'nullable|date_format:H:i',
            'modalidad' => 'required|in:presencial,virtual,telefonica',
            'tipo' => 'required|in:individual,grupal',
            'temas_tratados' => 'required|string|min:10|max:2000',
            'acuerdos' => 'nullable|string|max:2000',
            'compromisos_tutorado' => 'nullable|string|max:1000',
            'compromisos_tutor' => 'nullable|string|max:1000',
            'observaciones' => 'nullable|string|max:1000',
            'estado' => 'required|in:realizada,cancelada,pendiente',
            'proxima_sesion' => 'nullable|date|after:fecha',
        ];
    }

    public function openCreateModal(?int $tutoradoId = null): void
    {
        $this->resetForm();
        $this->fecha = now()->format('Y-m-d');
        if ($tutoradoId) {
            $this->tutorado_id = (string) $tutoradoId;
        }
        $this->editMode = false;
        $this->showModal = true;
    }

    public function openEditModal(int $sessionId): void
    {
        $session = TutoringSession::where('tutor_id', Auth::id())->findOrFail($sessionId);
        $this->editingId = $sessionId;
        $this->tutorado_id = (string) $session->tutorado_id;
        $this->fecha = $session->fecha->format('Y-m-d');
        $this->hora_inicio = $session->hora_inicio ?? '';
        $this->hora_fin = $session->hora_fin ?? '';
        $this->modalidad = $session->modalidad;
        $this->tipo = $session->tipo;
        $this->temas_tratados = $session->temas_tratados;
        $this->acuerdos = $session->acuerdos ?? '';
        $this->compromisos_tutorado = $session->compromisos_tutorado ?? '';
        $this->compromisos_tutor = $session->compromisos_tutor ?? '';
        $this->observaciones = $session->observaciones ?? '';
        $this->estado = $session->estado;
        $this->proxima_sesion = $session->proxima_sesion?->format('Y-m-d') ?? '';
        $this->editMode = true;
        $this->showModal = true;
    }

    public function save(): void
    {
        $this->validate();

        // Verificar que el tutorado pertenece a este tutor
        $asignacion = TutorAssignment::where('tutor_id', Auth::id())
            ->where('tutorado_id', $this->tutorado_id)
            ->where('activo', true)
            ->exists();

        if (!$asignacion) {
            $this->addError('tutorado_id', 'Este tutorado no está asignado a usted.');
            return;
        }

        $data = [
            'tutor_id' => Auth::id(),
            'tutorado_id' => $this->tutorado_id,
            'fecha' => $this->fecha,
            'hora_inicio' => $this->hora_inicio ?: null,
            'hora_fin' => $this->hora_fin ?: null,
            'modalidad' => $this->modalidad,
            'tipo' => $this->tipo,
            'temas_tratados' => $this->temas_tratados,
            'acuerdos' => $this->acuerdos ?: null,
            'compromisos_tutorado' => $this->compromisos_tutorado ?: null,
            'compromisos_tutor' => $this->compromisos_tutor ?: null,
            'observaciones' => $this->observaciones ?: null,
            'estado' => $this->estado,
            'proxima_sesion' => $this->proxima_sesion ?: null,
        ];

        if ($this->editMode) {
            TutoringSession::where('tutor_id', Auth::id())->findOrFail($this->editingId)->update($data);
            session()->flash('success', 'Sesión actualizada correctamente.');
        } else {
            TutoringSession::create($data);
            session()->flash('success', 'Sesión registrada correctamente.');
        }

        $this->showModal = false;
        $this->resetForm();
    }

    public function deleteSession(int $sessionId): void
    {
        TutoringSession::where('tutor_id', Auth::id())->findOrFail($sessionId)->delete();
        session()->flash('success', 'Sesión eliminada.');
    }

    protected function resetForm(): void
    {
        $this->tutorado_id = '';
        $this->fecha = '';
        $this->hora_inicio = '';
        $this->hora_fin = '';
        $this->modalidad = 'presencial';
        $this->tipo = 'individual';
        $this->temas_tratados = '';
        $this->acuerdos = '';
        $this->compromisos_tutorado = '';
        $this->compromisos_tutor = '';
        $this->observaciones = '';
        $this->estado = 'realizada';
        $this->proxima_sesion = '';
        $this->editingId = null;
        $this->resetErrorBag();
    }

    public function getMisTutoradosProperty()
    {
        return TutorAssignment::where('tutor_id', Auth::id())
            ->where('activo', true)
            ->with('tutorado')
            ->get()
            ->pluck('tutorado');
    }

    public function render()
    {
        $sessions = TutoringSession::where('tutor_id', Auth::id())
            ->when($this->filterTutorado, fn($q) => $q->where('tutorado_id', $this->filterTutorado))
            ->when($this->filterEstado, fn($q) => $q->where('estado', $this->filterEstado))
            ->when($this->filterMes, function ($q) {
                [$year, $month] = explode('-', $this->filterMes);
                $q->whereYear('fecha', $year)->whereMonth('fecha', $month);
            })
            ->with('tutorado')
            ->orderByDesc('fecha')
            ->paginate(15);

        return view('livewire.tutor.session-manager', [
            'sessions' => $sessions,
            'misTutorados' => $this->misTutorados,
        ])->layout('layouts.app', ['title' => 'Sesiones de Tutoría']);
    }
}
