<?php

namespace App\Http\Livewire\Admin;

use Livewire\Component;
use App\Models\User;
use App\Models\TutorAssignment;
use App\Models\TutoringSession;
use App\Models\ImportLog;
use Spatie\Activitylog\Models\Activity;

class Dashboard extends Component
{
    public int $totalTutores = 0;
    public int $totalTutorados = 0;
    public int $tutoradosSinAsignar = 0;
    public int $sesionesHoy = 0;
    public int $sesionesEstesMes = 0;
    public array $tutoradosPorTutor = [];
    public array $actividadReciente = [];
    public array $importacionesRecientes = [];
    public array $chartData = [];

    public function mount(): void
    {
        $this->cargarMetricas();
    }

    public function cargarMetricas(): void
    {
        $this->totalTutores = User::tutores()->activos()->count();
        $this->totalTutorados = User::tutorados()->activos()->count();
        $this->tutoradosSinAsignar = User::sinTutor()->activos()->count();
        $this->sesionesHoy = TutoringSession::whereDate('fecha', today())->count();
        $this->sesionesEstesMes = TutoringSession::whereMonth('fecha', now()->month)
            ->whereYear('fecha', now()->year)
            ->count();

        // Tutorados por tutor (top 10)
        $this->tutoradosPorTutor = User::tutores()
            ->activos()
            ->withCount(['tutoradosAsignados'])
            ->orderByDesc('tutor_assignments_count')
            ->limit(10)
            ->get()
            ->map(fn($t) => [
                'nombre' => $t->name,
                'total' => $t->tutor_assignments_count ?? 0,
            ])
            ->toArray();

        // Actividad reciente
        $this->actividadReciente = Activity::with('causer')
            ->latest()
            ->limit(10)
            ->get()
            ->map(fn($a) => [
                'descripcion' => $a->description,
                'usuario' => $a->causer?->name ?? 'Sistema',
                'fecha' => $a->created_at->diffForHumans(),
            ])
            ->toArray();

        // Importaciones recientes
        $this->importacionesRecientes = ImportLog::with(['importedBy', 'tutor'])
            ->latest()
            ->limit(5)
            ->get()
            ->map(fn($i) => [
                'archivo' => $i->filename,
                'por' => $i->importedBy?->name,
                'tutor' => $i->tutor?->name,
                'creados' => $i->created,
                'fecha' => $i->created_at->format('d/m/Y H:i'),
            ])
            ->toArray();

        // Datos para gráfica Chart.js
        $this->chartData = [
            'labels' => collect($this->tutoradosPorTutor)->pluck('nombre')->toArray(),
            'data' => collect($this->tutoradosPorTutor)->pluck('total')->toArray(),
        ];
    }

    public function render()
    {
        return view('livewire.admin.dashboard')
            ->layout('layouts.app', ['title' => 'Dashboard Administrador']);
    }
}
