<?php

namespace Database\Seeders;

use App\Models\User;
use App\Models\TutorAssignment;
use App\Models\TutoradoProfile;
use App\Models\TutoringSession;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class DatabaseSeeder extends Seeder
{
    public function run(): void
    {
        // =====================
        // CREAR ROLES
        // =====================
        $adminRole = Role::firstOrCreate(['name' => 'administrador', 'guard_name' => 'web']);
        $tutorRole = Role::firstOrCreate(['name' => 'tutor', 'guard_name' => 'web']);
        $tutoradoRole = Role::firstOrCreate(['name' => 'tutorado', 'guard_name' => 'web']);

        // =====================
        // PERMISOS GRANULARES
        // =====================
        $permissions = [
            // Usuarios
            'users.view', 'users.create', 'users.edit', 'users.delete',
            'users.export', 'users.import', 'users.reset-password',
            // Asignaciones
            'assignments.view', 'assignments.create', 'assignments.edit', 'assignments.delete',
            // Sesiones
            'sessions.view', 'sessions.create', 'sessions.edit', 'sessions.delete',
            // Bitácora
            'activitylog.view',
            // Configuración
            'config.view', 'config.edit',
        ];

        foreach ($permissions as $perm) {
            Permission::firstOrCreate(['name' => $perm, 'guard_name' => 'web']);
        }

        // Asignar permisos por rol
        $adminRole->syncPermissions($permissions); // Admin tiene todos
        $tutorRole->syncPermissions([
            'sessions.view', 'sessions.create', 'sessions.edit', 'sessions.delete',
            'assignments.view',
        ]);
        $tutoradoRole->syncPermissions([
            'sessions.view',
        ]);

        // =====================
        // CREAR ADMINISTRADOR
        // =====================
        $admin = User::firstOrCreate(
            ['email' => 'admin@universidad.edu'],
            [
                'name' => 'Administrador del Sistema',
                'codigo' => 'ADM001',
                'password' => Hash::make('Admin1234!'),
                'activo' => true,
                'email_verified_at' => now(),
            ]
        );
        $admin->assignRole('administrador');

        // =====================
        // CREAR TUTORES
        // =====================
        $tutoresData = [
            ['name' => 'Dr. Carlos Hernández López', 'email' => 'tutor1@universidad.edu', 'codigo' => 'TUT001'],
            ['name' => 'Mtra. Ana Martínez García', 'email' => 'tutor2@universidad.edu', 'codigo' => 'TUT002'],
            ['name' => 'Ing. Roberto Sánchez Torres', 'email' => 'tutor3@universidad.edu', 'codigo' => 'TUT003'],
        ];

        $tutores = [];
        foreach ($tutoresData as $td) {
            $tutor = User::firstOrCreate(
                ['email' => $td['email']],
                array_merge($td, [
                    'password' => Hash::make('Tutor1234!'),
                    'activo' => true,
                    'email_verified_at' => now(),
                    'carrera' => 'Ingeniería en Sistemas',
                ])
            );
            $tutor->assignRole('tutor');
            $tutores[] = $tutor;
        }

        // =====================
        // CREAR TUTORADOS
        // =====================
        $carreras = ['Ingeniería en Sistemas', 'Ingeniería Industrial', 'Contaduría Pública', 'Administración de Empresas'];
        $semestres = ['1', '2', '3', '4', '5', '6', '7', '8'];
        $estadosAcademicos = ['regular', 'regular', 'regular', 'irregular'];

        $tutoradosData = [
            ['name' => 'María González Pérez', 'email' => 'tutorado1@universidad.edu', 'codigo' => '20230001'],
            ['name' => 'Juan López Rodríguez', 'email' => 'tutorado2@universidad.edu', 'codigo' => '20230002'],
            ['name' => 'Laura Jiménez Flores', 'email' => 'tutorado3@universidad.edu', 'codigo' => '20230003'],
            ['name' => 'Pedro Martínez Cruz', 'email' => 'tutorado4@universidad.edu', 'codigo' => '20230004'],
            ['name' => 'Sofia Hernández Vargas', 'email' => 'tutorado5@universidad.edu', 'codigo' => '20230005'],
            ['name' => 'Diego Torres Mendoza', 'email' => 'tutorado6@universidad.edu', 'codigo' => '20230006'],
            ['name' => 'Valentina Ruiz Soto', 'email' => 'tutorado7@universidad.edu', 'codigo' => '20230007'],
            ['name' => 'Andrés Morales Castillo', 'email' => 'tutorado8@universidad.edu', 'codigo' => '20230008'],
            ['name' => 'Camila Reyes Ortega', 'email' => 'tutorado9@universidad.edu', 'codigo' => '20230009'],
            ['name' => 'Fernando García Lima', 'email' => 'tutorado10@universidad.edu', 'codigo' => '20230010'],
        ];

        foreach ($tutoradosData as $idx => $td) {
            $tutorado = User::firstOrCreate(
                ['email' => $td['email']],
                array_merge($td, [
                    'password' => Hash::make('Tut1234!'),
                    'activo' => true,
                    'email_verified_at' => now(),
                    'carrera' => $carreras[$idx % count($carreras)],
                    'semestre' => $semestres[$idx % count($semestres)],
                ])
            );
            $tutorado->assignRole('tutorado');

            // Crear perfil
            TutoradoProfile::firstOrCreate(
                ['user_id' => $tutorado->id],
                [
                    'numero_control' => $td['codigo'],
                    'carrera' => $carreras[$idx % count($carreras)],
                    'semestre' => $semestres[$idx % count($semestres)],
                    'turno' => $idx % 2 === 0 ? 'Matutino' : 'Vespertino',
                    'fecha_ingreso' => now()->subMonths(rand(6, 36)),
                    'estado_academico' => $estadosAcademicos[$idx % count($estadosAcademicos)],
                ]
            );

            // Asignar tutor (distribuir entre los 3 tutores)
            $tutorAsignado = $tutores[$idx % count($tutores)];
            TutorAssignment::firstOrCreate(
                ['tutorado_id' => $tutorado->id],
                [
                    'tutor_id' => $tutorAsignado->id,
                    'fecha_asignacion' => now()->subDays(rand(30, 180)),
                    'activo' => true,
                    'assigned_by' => $admin->id,
                ]
            );

            // Crear algunas sesiones de ejemplo
            if ($idx < 6) {
                TutoringSession::firstOrCreate(
                    ['tutor_id' => $tutorAsignado->id, 'tutorado_id' => $tutorado->id, 'fecha' => now()->subDays(rand(5, 60))],
                    [
                        'modalidad' => 'presencial',
                        'tipo' => 'individual',
                        'temas_tratados' => 'Revisión de avance académico, orientación sobre materias en riesgo.',
                        'acuerdos' => 'El tutorado se compromete a asistir regularmente a clases y buscar asesoría académica.',
                        'estado' => 'realizada',
                        'proxima_sesion' => now()->addDays(rand(7, 30)),
                    ]
                );
            }
        }

        $this->command->info('✅ Seeder ejecutado correctamente.');
        $this->command->info('');
        $this->command->info('Credenciales:');
        $this->command->info('  Admin:    admin@universidad.edu / Admin1234!');
        $this->command->info('  Tutor:    tutor1@universidad.edu / Tutor1234!');
        $this->command->info('  Tutorado: tutorado1@universidad.edu / Tut1234!');
    }
}
