<?php

namespace App\Http\Livewire\Admin;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\User;
use App\Models\TutorAssignment;
use App\Models\ImportLog;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Maatwebsite\Excel\HeadingRowImport;

class ImportTutorados extends Component
{
    use WithFileUploads;

    // Step management
    public int $step = 1; // 1: upload, 2: preview, 3: assign tutor, 4: results

    // File
    public $archivo = null;
    public string $archivoNombre = '';

    // Preview data
    public array $previewData = [];
    public array $headers = [];
    public int $totalFilas = 0;
    public array $erroresValidacion = [];
    public bool $previewValido = false;

    // Tutor assignment
    public string $tutorSeleccionado = '';
    public bool $confirmarImportacion = false;

    // Results
    public array $resultados = [];
    public array $credencialesGeneradas = [];
    public ?int $importLogId = null;

    // Loading state
    public bool $procesando = false;

    protected function rules(): array
    {
        return [
            'archivo' => 'required|file|mimes:csv,xlsx,xls|max:5120', // 5MB max
        ];
    }

    /**
     * Columnas esperadas en el archivo de importación
     */
    protected array $columnasRequeridas = ['nombre', 'email'];
    protected array $columnasOpcionales = ['codigo', 'telefono', 'carrera', 'semestre', 'numero_control'];

    public function updatedArchivo(): void
    {
        $this->validate(['archivo' => 'required|file|mimes:csv,xlsx,xls|max:5120']);
        $this->archivoNombre = $this->archivo->getClientOriginalName();
        $this->procesarPreview();
    }

    protected function procesarPreview(): void
    {
        try {
            $extension = $this->archivo->getClientOriginalExtension();
            $rows = [];

            if ($extension === 'csv') {
                $rows = $this->parseCsv();
            } else {
                $rows = $this->parseExcel();
            }

            if (empty($rows)) {
                $this->addError('archivo', 'El archivo está vacío o no se pudo leer.');
                return;
            }

            // Validar estructura
            $this->validarEstructura($rows);

            // Limitar preview a 100 filas
            $this->previewData = array_slice($rows, 0, 100);
            $this->totalFilas = count($rows);
            $this->step = 2;
        } catch (\Exception $e) {
            $this->addError('archivo', 'Error al procesar el archivo: ' . $e->getMessage());
        }
    }

    protected function parseCsv(): array
    {
        $path = $this->archivo->getRealPath();
        $rows = [];
        $headers = [];

        if (($handle = fopen($path, 'r')) !== false) {
            $lineNum = 0;
            while (($data = fgetcsv($handle, 1000, ',')) !== false) {
                $data = array_map('trim', $data);

                if ($lineNum === 0) {
                    $headers = array_map('strtolower', $data);
                    $this->headers = $headers;
                } else {
                    if (count($data) >= 2) {
                        $row = array_combine(
                            array_slice($headers, 0, count($data)),
                            $data
                        );
                        $rows[] = $row;
                    }
                }
                $lineNum++;
            }
            fclose($handle);
        }

        return $rows;
    }

    protected function parseExcel(): array
    {
        $rows = [];
        $data = Excel::toArray([], $this->archivo->getRealPath());

        if (empty($data) || empty($data[0])) {
            return [];
        }

        $sheet = $data[0];
        $headers = array_map('strtolower', array_map('trim', $sheet[0]));
        $this->headers = $headers;

        for ($i = 1; $i < count($sheet); $i++) {
            $rowData = $sheet[$i];
            if (array_filter($rowData)) { // Ignorar filas vacías
                $rows[] = array_combine(
                    array_slice($headers, 0, count($rowData)),
                    $rowData
                );
            }
        }

        return $rows;
    }

    protected function validarEstructura(array $rows): void
    {
        $this->erroresValidacion = [];
        $emailsEnArchivo = [];

        // Verificar columnas requeridas
        foreach ($this->columnasRequeridas as $col) {
            if (!in_array($col, $this->headers)) {
                $this->erroresValidacion[] = "Columna requerida faltante: '{$col}'";
            }
        }

        if (!empty($this->erroresValidacion)) {
            $this->previewValido = false;
            return;
        }

        // Validar datos fila por fila
        foreach ($rows as $idx => $row) {
            $fila = $idx + 2; // +2 porque idx 0 = fila 2 (fila 1 es header)

            if (empty($row['nombre'])) {
                $this->erroresValidacion[] = "Fila {$fila}: El nombre es requerido.";
            }

            if (empty($row['email'])) {
                $this->erroresValidacion[] = "Fila {$fila}: El email es requerido.";
            } elseif (!filter_var($row['email'], FILTER_VALIDATE_EMAIL)) {
                $this->erroresValidacion[] = "Fila {$fila}: Email inválido '{$row['email']}'.";
            } elseif (in_array($row['email'], $emailsEnArchivo)) {
                $this->erroresValidacion[] = "Fila {$fila}: Email duplicado en el archivo '{$row['email']}'.";
            } else {
                $emailsEnArchivo[] = $row['email'];
            }
        }

        $this->previewValido = empty($this->erroresValidacion) || count($this->erroresValidacion) < 5;
    }

    public function avanzarASeleccionTutor(): void
    {
        if (!$this->previewValido && count($this->erroresValidacion) >= 5) {
            session()->flash('error', 'El archivo tiene demasiados errores. Corrígelos antes de continuar.');
            return;
        }
        $this->step = 3;
    }

    public function regresarAPreview(): void
    {
        $this->step = 2;
    }

    public function regresarAUpload(): void
    {
        $this->step = 1;
        $this->reset(['archivo', 'previewData', 'headers', 'totalFilas', 'erroresValidacion', 'previewValido', 'tutorSeleccionado']);
    }

    public function confirmar(): void
    {
        $this->validate([
            'tutorSeleccionado' => 'required|exists:users,id',
        ]);
        $this->confirmarImportacion = true;
    }

    public function ejecutarImportacion(): void
    {
        $this->validate([
            'tutorSeleccionado' => 'required|exists:users,id',
        ]);

        $this->procesando = true;

        $tutor = User::findOrFail($this->tutorSeleccionado);

        $resultados = [
            'creados' => 0,
            'actualizados' => 0,
            'omitidos' => 0,
            'errores' => 0,
        ];
        $erroresDetalle = [];
        $this->credencialesGeneradas = [];

        // Re-parsear el archivo completo para la importación real
        $extension = $this->archivo->getClientOriginalExtension();
        $allRows = $extension === 'csv' ? $this->parseCsv() : $this->parseExcel();

        DB::beginTransaction();
        try {
            foreach ($allRows as $idx => $row) {
                $fila = $idx + 2;
                try {
                    if (empty($row['email']) || !filter_var($row['email'], FILTER_VALIDATE_EMAIL)) {
                        $resultados['omitidos']++;
                        $erroresDetalle[] = "Fila {$fila}: Email inválido, omitida.";
                        continue;
                    }

                    $existingUser = User::where('email', $row['email'])->first();

                    if ($existingUser) {
                        // Actualizar datos básicos
                        $existingUser->update([
                            'name' => $row['nombre'] ?? $existingUser->name,
                            'codigo' => $row['codigo'] ?? $existingUser->codigo,
                            'telefono' => $row['telefono'] ?? $existingUser->telefono,
                        ]);

                        // Actualizar/crear asignación de tutor
                        TutorAssignment::updateOrCreate(
                            ['tutorado_id' => $existingUser->id],
                            [
                                'tutor_id' => $tutor->id,
                                'fecha_asignacion' => now(),
                                'activo' => true,
                                'assigned_by' => Auth::id(),
                            ]
                        );

                        $resultados['actualizados']++;
                    } else {
                        // Crear nuevo usuario
                        $tempPassword = $this->generarPasswordTemporal();

                        $newUser = User::create([
                            'name' => $row['nombre'],
                            'email' => $row['email'],
                            'codigo' => $row['codigo'] ?? null,
                            'telefono' => $row['telefono'] ?? null,
                            'carrera' => $row['carrera'] ?? null,
                            'semestre' => $row['semestre'] ?? null,
                            'password' => Hash::make($tempPassword),
                            'activo' => true,
                        ]);

                        $newUser->assignRole('tutorado');

                        // Asignar tutor
                        TutorAssignment::create([
                            'tutor_id' => $tutor->id,
                            'tutorado_id' => $newUser->id,
                            'fecha_asignacion' => now(),
                            'activo' => true,
                            'assigned_by' => Auth::id(),
                        ]);

                        // Crear perfil básico
                        $newUser->perfil()->create([
                            'numero_control' => $row['numero_control'] ?? null,
                            'carrera' => $row['carrera'] ?? null,
                            'semestre' => $row['semestre'] ?? null,
                        ]);

                        $this->credencialesGeneradas[] = [
                            'nombre' => $newUser->name,
                            'email' => $newUser->email,
                            'codigo' => $newUser->codigo ?? 'N/A',
                            'password_temporal' => $tempPassword,
                        ];

                        $resultados['creados']++;
                    }
                } catch (\Exception $e) {
                    $resultados['errores']++;
                    $erroresDetalle[] = "Fila {$fila}: " . $e->getMessage();
                }
            }

            // Guardar log de importación
            $importLog = ImportLog::create([
                'imported_by' => Auth::id(),
                'tutor_id' => $tutor->id,
                'filename' => $this->archivoNombre,
                'total_rows' => count($allRows),
                'created' => $resultados['creados'],
                'updated' => $resultados['actualizados'],
                'skipped' => $resultados['omitidos'],
                'errors' => $resultados['errores'],
                'error_details' => $erroresDetalle,
                'credentials_generated' => $this->credencialesGeneradas,
                'status' => 'completed',
            ]);

            $this->importLogId = $importLog->id;

            activity()->causedBy(Auth::user())
                ->log("Importación masiva: {$resultados['creados']} creados, {$resultados['actualizados']} actualizados para tutor {$tutor->name}");

            DB::commit();

            $this->resultados = $resultados;
            $this->step = 4;
        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash('error', 'Error durante la importación: ' . $e->getMessage());
        }

        $this->procesando = false;
        $this->confirmarImportacion = false;
    }

    protected function generarPasswordTemporal(): string
    {
        $chars = 'abcdefghijklmnopqrstuvwxyz';
        $upper = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $digits = '0123456789';

        return $upper[rand(0, 25)]
            . substr(str_shuffle($chars), 0, 4)
            . $digits[rand(0, 9)]
            . $digits[rand(0, 9)]
            . '!';
    }

    public function nuevaImportacion(): void
    {
        $this->reset([
            'archivo', 'archivoNombre', 'previewData', 'headers', 'totalFilas',
            'erroresValidacion', 'previewValido', 'tutorSeleccionado',
            'confirmarImportacion', 'resultados', 'credencialesGeneradas',
            'importLogId', 'procesando',
        ]);
        $this->step = 1;
    }

    public function getTutoresProperty()
    {
        return User::tutores()->activos()->orderBy('name')->get();
    }

    public function render()
    {
        return view('livewire.admin.import-tutorados', [
            'tutores' => $this->tutores,
        ])->layout('layouts.app', ['title' => 'Importación Masiva de Tutorados']);
    }
}
