<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Jetstream\HasProfilePhoto;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Traits\HasRoles;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, HasProfilePhoto, Notifiable, HasRoles, SoftDeletes, LogsActivity;

    protected $fillable = [
        'name',
        'codigo',
        'email',
        'password',
        'telefono',
        'carrera',
        'semestre',
        'activo',
    ];

    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_recovery_codes',
        'two_factor_secret',
    ];

    protected $appends = [
        'profile_photo_url',
    ];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'activo' => 'boolean',
        ];
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['name', 'email', 'codigo', 'activo'])
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs();
    }

    // =====================
    // RELATIONSHIPS
    // =====================

    /**
     * Como Tutor: tutorados asignados
     */
    public function tutoradosAsignados()
    {
        return $this->hasMany(TutorAssignment::class, 'tutor_id')->where('activo', true);
    }

    /**
     * Como Tutorado: tutor asignado
     */
    public function tutorAsignado()
    {
        return $this->hasOne(TutorAssignment::class, 'tutorado_id')->where('activo', true);
    }

    /**
     * Como Tutor: todas las asignaciones (historial)
     */
    public function todasLasAsignaciones()
    {
        return $this->hasMany(TutorAssignment::class, 'tutor_id');
    }

    /**
     * Perfil del tutorado
     */
    public function perfil()
    {
        return $this->hasOne(TutoradoProfile::class);
    }

    /**
     * Sesiones como tutor
     */
    public function sesionesComoTutor()
    {
        return $this->hasMany(TutoringSession::class, 'tutor_id');
    }

    /**
     * Sesiones como tutorado
     */
    public function sesionesComoTutorado()
    {
        return $this->hasMany(TutoringSession::class, 'tutorado_id');
    }

    // =====================
    // SCOPES
    // =====================

    public function scopeActivos($query)
    {
        return $query->where('activo', true);
    }

    public function scopeInactivos($query)
    {
        return $query->where('activo', false);
    }

    public function scopeTutores($query)
    {
        return $query->role('tutor');
    }

    public function scopeTutorados($query)
    {
        return $query->role('tutorado');
    }

    public function scopeSinTutor($query)
    {
        return $query->role('tutorado')
            ->whereDoesntHave('tutorAsignado');
    }

    // =====================
    // HELPERS
    // =====================

    public function esTutor(): bool
    {
        return $this->hasRole('tutor');
    }

    public function esTutorado(): bool
    {
        return $this->hasRole('tutorado');
    }

    public function esAdmin(): bool
    {
        return $this->hasRole('administrador');
    }

    public function getTutorAttribute(): ?User
    {
        return $this->tutorAsignado?->tutor;
    }

    public function getRolDisplayAttribute(): string
    {
        return match($this->roles->first()?->name) {
            'administrador' => 'Administrador',
            'tutor' => 'Tutor',
            'tutorado' => 'Tutorado',
            default => 'Sin rol',
        };
    }
}
